module net.BurtonRadons.dig.platform.bitmap;

//private import std.c.windows.windows;


private extern (C) int qsort (void *p, int eachsize, int count, int (*decl) (void *, void *));

/** A 2d matrix of RGBA values. */
class Bitmap
{
    private import net.BurtonRadons.dig.platform.base;
    private import net.BurtonRadons.dig.common.brush;
    private import net.BurtonRadons.dig.platform.canvas;
    private import net.BurtonRadons.dig.platform.control;
    private import net.BurtonRadons.dig.platform.frame;
    private import net.BurtonRadons.dig.platform.windows;
    
    /** A raster transfer mode. */
    enum RasterOp
    {
        Copy, /**< Copy colors between the source and the destination (destination = source). */
        XOR, /**< Binary exclusive-or the colors in the color-space (destination = destination ^ source). */
    }
    
    const int bytesPerPixel = 4; /**< The number of bytes to a pixel. */

    Contorter contort; /**< The contorter to use for polygon points and pixels rendered to the bitmap. */

    /** Create a bitmap.  Make it compatible with the Control and have the specified dimensions. */
    this (int width, int height)
    {
        digPlatformWidth = width;
        digPlatformHeight = height;
        digPlatformHDC = CreateCompatibleDC (null);

        _BITMAPINFO info;
        _BITMAPINFOHEADER *head = &info.bmiHeader;

        head.biSize = _BITMAPINFOHEADER.size;
        head.biWidth = width;
        head.biHeight = height;
        head.biPlanes = 1;
        head.biBitCount = 32;
        head.biCompression = BI_RGB;
        head.biSizeImage = 0;
        head.biXPelsPerMeter = 100;
        head.biYPelsPerMeter = 100;
        head.biClrUsed = 0;
        head.biClrImportant = 0;

        digPlatformBitmap = CreateDIBSection (digPlatformHDC, &info, DIB_RGB_COLORS, (void **) &digPlatformData, (_HANDLE) 0, 0);
        SelectObject (digPlatformHDC, digPlatformBitmap);
    }

    ~this ()
    {
        DeleteDC (digPlatformHDC);
        DeleteObject (digPlatformBitmap);
    }

    /** Determine whether this point is in-range. */
    final bit inRange (int x, int y)
    {
        return x >= 0 && y >= 0 && x < digPlatformWidth && y < digPlatformHeight;
    }

    /** Determine whether the horizontal coordinate is in-range. */
    final bit inRangeX (int x)
    {
        return x >= 0 && x < digPlatformWidth;
    }

    /** Determine whether the vertical coordinate is in-range. */
    final bit inRangeY (int y)
    {
        return y >= 0 && y < digPlatformHeight;
    }

    /** Read a pixel from the bitmap.  If out-of-range, this returns black. */
    final Color get (int x, int y)
    {
        if (!inRange (x, y))
            return Color.Black;
        else
            return retrieve (row (x, y));
    }

    /** Read a pixel from the bitmap, assuming it's in-range. */
    final Color getx (int x, int y)
    {
        return retrieve (row (x, y));
    }

    /** Store a pixel in the bitmap.
      * If it's out-of-range, it is ignored.
      * This blends the pixel into the current color.
      */

    final void set (int x, int y, Color c)
    {
        if (!inRange (x, y))
            return;

        ubyte *d = row (x, y);
        int a = c.a;
        int n = 255 - a;

        if (a == 255)
        {
            d [2] = c.r;
            d [1] = c.g;
            d [0] = c.b;
            d [3] = 255;
        }
        else
        {
            d [2] = (d [2] * n + c.r * a) / 255;
            d [1] = (d [1] * n + c.g * a) / 255;
            d [0] = (d [0] * n + c.b * a) / 255;
            d [3] = 255 - (255 - d [3]) * (255 - a) / 255;
        }
    }

    /** Set a horizontal row on the bitmap, no blending. */
    final void hseto (int sx, int y, Color [] colors)
    {
        if (y < 0 || y >= digPlatformHeight || sx >= digPlatformWidth)
            return;

        if (colors.length + sx > digPlatformWidth)
            colors = colors [0 .. digPlatformWidth - sx];
        if (sx < 0)
        {
            if (-sx >= colors.length)
                return;
            colors = colors [-sx .. colors.length];
            sx = 0;
        }

        ubyte *d = row (sx, y);

        for (Color *c = colors, e = c + colors.length; c < e; c ++, d += bytesPerPixel)
        {
            d [2] = c.r;
            d [1] = c.g;
            d [0] = c.b;
            d [3] = c.a;
        }
    }

    /** The width of the bitmap in pixels. */
    final int width () { return digPlatformWidth; }

    /** The height of the bitmap in pixels. */
    final int height () { return digPlatformHeight; }

/* .. Drawing functions ... */

    /** Clear to a std.math.single color. */
    final void clear (Color c)
    {
        ubyte *d = digPlatformData;
        ubyte *e = digPlatformData + digPlatformWidth * digPlatformHeight * bytesPerPixel;

        for ( ; d < e; d += bytesPerPixel)
        {
            d [0] = c.b;
            d [1] = c.g;
            d [2] = c.r;
            d [3] = c.a;
        }
    }

    /** Draw a rectangle on the bitmap using pen around the edges and brush inside. */
    final void rect (int sx, int sy, int ex, int ey, Brush brush)
    {
        if (ex < sx) { int tx = sx; sx = ex; ex = tx; }
        if (ey < sy) { int ty = sy; sy = ey; ey = ty; }
        bound (sx, sy);
        bound (ex, ey);

        for (int x = sx; x <= ex; x ++)
        {
            for (int y = sy; y <= ey; y ++)
                set (x, y, brush.eval (x, y));
        }
    }

    /** Draw an outline rectangle, inclusive. */
    final void outlineRect (int sx, int sy, int ex, int ey, Brush brush)
    {
        int t;

        if (sx > ex) t = sx, sx = ex, ex = t;
        if (sy > ey) t = sy, sy = ey, ey = t;

        if (sx == ex)
            vline (sx, sy, ey, brush);
        else if (sy == ey)
            hline (sx, sy, ex, brush);
        else
        {
            hline (sx, sy, ex, brush);
            hline (sx, ey, ex, brush);
            vline (sx, sy + 1, ey - 1, brush);
            vline (ex, sy + 1, ey - 1, brush);
        }
    }

    /** Draw a horizontal line, inclusive. */
    final void hline (int sx, int y, int ex, Brush brush)
    {
        if (sx > ex) { int tx = sx; sx = ex; ex = tx; }
        if (y < 0 || y >= digPlatformHeight)
            return;
        boundx (sx);
        boundx (ex);

        for (int x = sx; x <= ex; x ++)
            set (x, y, brush.eval (x, y));
    }

    /** Draw a vertical line, inclusive. */
    final void vline (int x, int sy, int ey, Brush brush)
    {
        if (sy > ey) { int ty = sy; sy = ey; ey = ty; }
        if (x < 0 || x >= digPlatformWidth)
            return;
        boundy (sy);
        boundy (ey);

        for (int y = sy; y <= ey; y ++)
            set (x, y, brush.eval (x, y));
    }

    void arect (int sx, int sy, int ex, int ey, Color color)
    {
        if (ex < sx) { int tx = sx; sx = ex; ex = tx; }
        if (ey < sy) { int ty = sy; sy = ey; ey = ty; }
        bound (sx, sy);
        bound (ex, ey);

        for (int y = sy; y <= ey; y ++)
        {
            ubyte *d = this.row (sx, y);
            ubyte *e = d + (ex - sx + 1) * bytesPerPixel;

            while (d < e)
            {
                store (d, color);
                d += bytesPerPixel;
            }
        }
    }

    /** A polygon edge and scan state. */
    struct Segment
    {
        float sx; /**< Starting horizontal coordinate of the segment. */
        float sy; /**< Starting vertical coordinate of the segment (must be at most ey). */
        float ex; /**< Ending horizontal coordinate of the segment. */
        float ey; /**< Ending horizontal coordinate of the segment (must be at least sy). */
        bit up; /**< Whether this is segment is up (true) or down (false). */

        float px; /**< Starting point of this scan. */
        
        /** Store variables as appropriate.  sy can be above ey; if it is,
          * they're swapped and up is set to true.
          * miny is set to sy if it is above it or if miny is float.nan.
          * maxy is set to ey if it is below it or if maxy is float.nan.
          */

        void setup (float sx, float sy, float ex, float ey, inout float miny, inout float maxy)
        {
            if (sy < ey)
            {
                this.sx = sx; this.sy = sy;
                this.ex = ex; this.ey = ey;
            }
            else
            {
                this.sx = ex; this.sy = ey;
                this.ex = sx; this.ey = sy;
                up = true;
            }
            
            if (std.math.isnan (miny) || sy < miny) miny = sy;
            if (std.math.isnan (maxy) || ey > maxy) maxy = ey;
        }

        /** Evaluate the x coordinate at the given y point.  Out-of-range
          * values are saturated.
          */

        float evaly (float y)
        {
            if (y <= sy) return sx;
            if (y >= ey) return ex;
            return (ex - sx) * (y - sy) / (ey - sy) + sx;
        }

        /** Setup parameters for a given vertical scanline.
          * minx is set to px if it is above it or if minx is float.nan.
          * maxx is set to qx if it is above it or if maxx is float.nan.
          */
        
        void pointy (float y, inout float minx, inout float maxx)
        {
            px = evaly (y);
            if (std.math.isnan (minx) || px < minx) minx = px;
            if (std.math.isnan (maxx) || px > maxx) maxx = px;
        }
    };

/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/

    extern (C) static int compareSegment (void *a, void *b)
    {
        float ax = ((Segment *) a).px;
        float bx = ((Segment *) b).px;

        if (ax < bx) return -1;
        if (ax > bx) return 1;
        return 0;
    }

/+
#endif
+/

    /** Draw a filled polygon using a std.math.single loop.  This uses
      * odd/even fill mode; a star drawn using five points will
      * have a hollowed-out center.
      */

    final void polygon (float p [], Brush brush)
    {
        Segment [] segs;
        Segment [] ints;
        float sy, ey;

        segs.length = p.length / 2;
        ints.length = segs.length;
        for (int c; c < p.length; c += 2)
        {
            Segment *s = &segs [c / 2];

            s.setup (p [c], p [c + 1], p [(c + 2) % p.length], p [(c + 3) % p.length], sy, ey);
        }

        if (sy < 0) sy = 0;
        if (ey >= digPlatformHeight) ey = digPlatformHeight - 1;

        for (int y = sy; y <= ey; y ++)
        {
            float minx, maxx;
            int n = 0;
            int l = 0;

            for (int c; c < segs.length; c ++)
            {
                if (segs [c].sy > y || segs [c].ey <= y)
                    continue;
                ints [n ++] = segs [c];
                ints [n - 1].pointy (y, minx, maxx);
            }

            if (n == 0)
                continue;

            if (minx < 0) minx = 0;
            if (maxx >= digPlatformWidth) maxx = digPlatformWidth - 1;

            qsort ((void *) ints, n, Segment.size, &compareSegment);

            for (int x = minx; x <= maxx; x ++)
                set (x, y, brush.eval (x, y));
        }

        delete segs;
        delete ints;
    }

    /** Draw a line on the bitmap using the current pen.
      * This creates square caps on the ends of the line.
      *
      * @param sx The horizontal starting coordinate of the line.
      * @param sy The vertical starting coordinate of the line.
      * @param ex The horizontal ending coordinate of the line.
      * @param ey The vertical ending coordinate of the line.
      * @param brush The brush to use for fill and pen parameters.
      */

    final void line (float sx, float sy, float ex, float ey, Brush brush)
    {
        float dx = ex - sx, dy = ey - sy;
        float d = std.math.sqrt (dx * dx + dy * dy);
        float nx = dx / d, ny = dy / d;
        float p = brush.width / 2.0;

        float [8] points;

        points [0] = sx - ny * p;
        points [1] = sy + nx * p;

        points [2] = ex - ny * p;
        points [3] = ey + nx * p;

        points [4] = ex + ny * p;
        points [5] = ey - nx * p;

        points [6] = sx + ny * p;
        points [7] = sy - nx * p;

        polygon (points, brush);
    }

    /** Blit the bitmap onto the canvas.  The canvas must be in between a
      * beginPaint/endPaint pair.  The alpha channel is ignored in the
      * blit, which means that if the bitmap is partially transparent the
      * colors could come out oddly (for example, objects drawn on it that
      * had no visual effect will show up).  The solution is to either
      * preprocess the bitmap or blit it onto another bitmap first.
      *
      * @param canvas The canvas to blit onto.
      * @param x The horizontal coordinate to blit onto.
      * @param y The vertical coordinate to blit onto.
      */

    void blit (Canvas canvas, int x, int y)
    {
        blit (canvas, x, y, 0, 0, digPlatformWidth, digPlatformHeight);
    }

    /** Blit the bitmap onto the canvas.  The canvas must be in between a
      * beginPaint/endPaint pair.  The alpha channel is ignored in the
      * blit, which means that if the bitmap is partially transparent the
      * colors could come out oddly (for example, objects drawn on it that
      * had no visual effect will show up).  The solution is to either
      * preprocess the bitmap or blit it onto another bitmap first.
      *
      * @param canvas The canvas to blit onto.
      * @param x The horizontal coordinate to blit onto.
      * @param y The vertical coordinate to blit onto.
      * @param mode The raster transfer mode to use.
      */

    void blit (Canvas canvas, int x, int y, RasterOp mode)
    {
        blit (canvas, x, y, 0, 0, digPlatformWidth, digPlatformHeight, mode);
    }
    
    
    /** Blit the bitmap onto the canvas, specifying the region to blit from.
      * The canvas must be in between a beginPaint/endPaint pair.  The
      * alpha channel is ignored in the blit, which means if the bitmap is
      * partially transparent the colors could come out oddly (for example,
      * objects drawn on it that had no visual effect will show up).  The
      * solution is to either preprocess the bitmap or blit it onto another
      * bitmap first.
      *
      * @param canvas The canvas to blit onto.
      * @param x The horizontal coordinate to blit onto.
      * @param y The vertical coordinate to blit onto.
      * @param sourcex The horizontal start of the rectangle on the bitmap to blit from.
      * @param sourcey The vertical start of the rectangle on the bitmap to blit from.
      * @param width The width in pixels to blit from.
      * @param height The height in pixels to blit from.
      */
      
    void blit (Canvas canvas, int x, int y, int sourcex, int sourcey, int width, int height)
    {
        blit (canvas, x, y, sourcex, sourcey, width, height, RasterOp.Copy);
    }
    
    /** Blit the bitmap onto the canvas, specifying the region to blit from.
      * The canvas must be in between a beginPaint/endPaint pair.  The
      * alpha channel is ignored in the blit, which means if the bitmap is
      * partially transparent the colors could come out oddly (for example,
      * objects drawn on it that had no visual effect will show up).  The
      * solution is to either preprocess the bitmap or blit it onto another
      * bitmap first.
      *
      * @param canvas The canvas to blit onto.
      * @param x The horizontal coordinate to blit onto.
      * @param y The vertical coordinate to blit onto.
      * @param sourcex The horizontal start of the rectangle on the bitmap to blit from.
      * @param sourcey The vertical start of the rectangle on the bitmap to blit from.
      * @param width The width in pixels to blit from.
      * @param height The height in pixels to blit from.
      * @param mode The raster transfer mode to use.
      */

    void blit (Canvas canvas, int x, int y, int sourcex, int sourcey, int width, int height, RasterOp mode)
    {
        digPlatformBaseBlit (canvas.digPlatformHDC, x, y, sourcex, sourcey, width, height, mode);
    } 
    
    /** Blit contents of this bitmap onto the canvas while stretching its dimensions.
      * @param canvas The canvas to blit onto.
      * @param dx The starting horizontal coordinate to blit onto.
      * @param dy The starting vertical coordinate to blit onto.
      * @param dw The width in pixels to render the bitmap as.
      * @param dh The height in pixels to render the bitmap as.
      * @param sx The horizontal coordinate in the bitmap to start blitting from.
      * @param sy The vertical coordinate in the bitmap to start blitting from.
      * @param sw The width in pixels of the region to blit from.
      * @param sh The height in pixels of the region to blit from.
      */
      
    void stretchBlit (Canvas canvas, int dx, int dy, int dw, int dh, int sx, int sy, int sw, int sh)
    {
        std.c.windows.windows.StretchBlt (canvas.digPlatformHDC, dx, dy, dw, dh, digPlatformHDC, sx, sy, sw, sh, SRCCOPY);
    }
    
    /** Blit the entire contents of this bitmap onto the canvas while stretching its dimensions.
      * @param dx The starting horizontal coordinate to blit onto.
      * @param dy The starting vertical coordinate to blit onto.
      * @param dw The width in pixels to render the bitmap as.
      * @param dh The height in pixels to render the bitmap as.
      */
      
    void stretchBlit (Canvas canvas, int dx, int dy, int dw, int dh)
    {
        stretchBlit (canvas, dx, dy, dw, dh, 0, 0, width (), height ());
    }

    /** Blit contents of this bitmap on another bitmap while stretching its
      * dimensions.  This chooses the nearest point.  This is currently poorly
      * implemented and will be very slow.
      *
      * @param dest The bitmap to blit onto.
      * @param dx The horizontal coordinate to blit onto.
      * @param dy The vertical coordinate to blit onto.
      * @param dw The width in pixels of the region to blit onto.
      * @param dh The height in pixels of the region to blit onto.
      * @param sx The horizontal coordinate to start blitting from.
      * @param sy The vertical coordinate to start blitting from.
      * @param sw The width in pixels of the region to blit from.
      * @param sh The height in pixels of the region to blit from.
      */

    void stretchBlit (Bitmap dest, int dx, int dy, int dw, int dh, int sx, int sy, int sw, int sh)
    {
        std.c.windows.windows.StretchBlt (dest.digPlatformHDC, dx, dy, dw, dh, digPlatformHDC, sx, sy, sw, sh, SRCCOPY);
        return;
        Bitmap src = this;

        for (int y; y < dh; y ++)
        {
            int dpy = y + dy;
            int spy = sy + y * sh / dh;

            if (!dest.inRangeY (dpy) || !src.inRangeY (spy))
                continue;
            for (int x; x < dw; x ++)
            {
                int dpx = x + dx;
                int spx = sx + x * sw / dw;

                if (!dest.inRangeX (dpx) || !src.inRangeX (spx))
                    continue;
                ubyte* dp = dest.row (dpx, dpy);
                ubyte* sp = src.row (spx, spy);

                dp [0 .. 4] = sp [0 .. 4];
            }
        }
    }

    protected final
    void circlePoint (inout float [] points,
                      float px, float py,
                      float x, float y,
                      float w, float h, float a, float b)
    {
        float lx = points.length ? points [points.length - 2] : std.math.sin (a) * w + x;
        float ly = points.length ? points [points.length - 1] : std.math.cos (a) * h + y;
        if (!points.length)
            contort.eval (lx, ly);

        float dx = px - lx;
        float dy = py - ly;
        float d = std.math.sqrt (dx * dx + dy * dy);

        if (d > 1)
        {
            float m = (b - a) / 2 + a;
            float rx = std.math.sin (m) * w + x;
            float ry = std.math.cos (m) * h + y;

            contort.eval (rx, ry);
            circlePoint (points, rx, ry, x, y, w, h, a, m);
            points ~= rx;
            points ~= ry;
            circlePoint (points, px, py, x, y, w, h, m, b);
        }
        else
        {
            points ~= px;
            points ~= py;
        }
    }

    /** Call circle (x, y, radius, radius, brush). */
    final void circle (float x, float y, float radius, Brush brush)
    {
        circle (x, y, radius, radius, brush);
    }

    /** Draw an ellipse on the bitmap.
      *
      * @param x The center horizontal coordinate.
      * @param y The vertical horizontal coordinate.
      * @param w The horizontal radius of the ellipse.
      * @param h The vertical radius of the ellipse.
      * @param brush The brush to use for coloring.
      */

    final void circle (float x, float y, float w, float h, Brush brush)
    {
        int count = 8;
        float [] points;

        for (int c; c < count; c ++)
        {
            float px = std.math.sin (2 * std.math.PI * c / count) * w + x;
            float py = std.math.cos (2 * std.math.PI * c / count) * h + y;

            contort.eval (px, py);
            circlePoint (points, px, py, x, y, w, h, 2 * std.math.PI * (c - 1) / count, 2 * std.math.PI * c / count);
        }

        polygon (points, brush);
        delete points;
    }

    /** Store a color into a data pointer. */
    static void store (ubyte *data, Color color)
    {
        data [0] = color.b;
        data [1] = color.g;
        data [2] = color.r;
        data [3] = color.a;
    }

    /** Store a color into a data point and increment the pointer. */
    static void storeinc (inout ubyte *data, Color color)
    {
        store (data, color);
        data += bytesPerPixel;
    }

    /** Retrieve a color from a data pointer. */
    static Color retrieve (ubyte *data)
    {
        return AColor (data [2], data [1], data [0], data [3]);
    }

    /** Retrieve a color from a data pointer and increment the pointer. */
    static Color retrieveinc (inout ubyte *data)
    {
        data += bytesPerPixel;
        return retrieve (data - bytesPerPixel);
    }

    /** Return pointer to this row's data.  Use #store and #retrieve to access the values.  Does NO bounds checking. */
    final ubyte *row (int x, int y)
    {
        return digPlatformData + (digPlatformHeight - y - 1) * digPlatformWidth * bytesPerPixel + x * bytesPerPixel;
    }

    /** If x is out-of-range, saturate it to the edges. */
    final void boundx (inout int x) { x = x < 0 ? 0 : x >= digPlatformWidth ? digPlatformWidth - 1 : x; }
    
    /** If y is out-of-range, saturate it to the edges. */
    final void boundy (inout int y) { y = y < 0 ? 0 : y >= digPlatformHeight ? digPlatformHeight - 1 : y; }
    
    /** If x or y are out-of-range, saturate them to the edges. */
    final void bound (inout int x, inout int y) { boundx (x); boundy (y); }
    
/+
#ifdef DoxygenMustSkipThis
+/

    _HDC digPlatformHDC;
    _HBITMAP digPlatformBitmap;
    ubyte *digPlatformData; /* RGB triplets */
    int digPlatformWidth;
    int digPlatformHeight;
    
    void digPlatformBaseBlit (_HDC destHDC, int x, int y, int sourcex, int sourcey, int width, int height, RasterOp mode)
    {
        int actualMode;
        
        switch (mode)
        {
            case RasterOp.Copy: actualMode = SRCCOPY; break;
            case RasterOp.XOR: actualMode = 0x00660046; break;
        }
                
        BitBlt (destHDC, x, y, width, height, digPlatformHDC, sourcex, sourcey, actualMode);
    }
    
/+
#endif
+/
}
